# Copyright(c) 1986 Association of Universities for Research in Astronomy Inc.

include	<mach.h>
include	<time.h>
include	<fset.h>
include <mii.h>

#------------------------------------------------------------------------------
# FITSSPP.X - IRAF/SPP interface for FITSIO.
#
#	ftopnx - open a FITS file
#	ftclsx - close a FITS file
#	ftpi2b - write an array of integer*2 bytes 
#	ftpi4b - write an array of integer*4 bytes 
#	ftpr4b - write an array of real*4 bytes 
#	ftpr8b - write an array of real*8 bytes 
#	ftgi2b - read an array of integer*2 bytes 
#	ftgi4b - read an array of integer*4 bytes 
#	ftgr4b - read an array of real*4 bytes 
#	ftgr8b - read an array of real*8 bytes 
#       ftpbyx - write a record to a FITS file
#       fsupk  - unpack Fortran characters into an SPP CHAR array.
#       ftgbyx - read a record from a FITS file
#	ftgsdt - get the current date
#	ftupch - convert a string to upper case
#	ftpbyt - append a string of bytes to a common block buffer 
#	ftgbyt - read a string of bytes from a common block buffer 
#
# These routines are part of the FITSIO library and are designed to run in
# the IRAF/SPP environment.
#------------------------------------------------------------------------------
#   This software was prepared by High Energy Astrophysic Science Archive
#   Research Center (HEASARC) at the NASA Goddard Space Flight Center. Users
#   shall not, without prior written permission of the U.S. Government,
#   establish a claim to statutory copyright.  The Government and others acting
#   on its behalf, shall have a royalty-free, non-exclusive, irrevocable,
#   worldwide license for Government purposes to publish, distribute,
#   translate, copy, exhibit, and perform such material. 
#------------------------------------------------------------------------------

define	SZ_FITSREC	1440		# FITS record size in chars


# FTOPNX -- Open or create a file.

procedure ftopnx (funit, pkname, oldnew, rwmode, block, status)

int	funit		#I Fortran I/O unit number
%       character*(*) pkname  
int	oldnew		#I file status: 0 =  existing file; else  new file
int	rwmode		#I file access mode: 0 = readonly; else = read/write
int	block		#O FITS record blocking factor 
int	status		#U returned error status (0=ok)

bool	firsttime
int	mode, i, ibuff, fd
char    fname[SZ_PATHNAME]
int	access(), open()
int	fstati()
data	firsttime /true/
include	"fitsspp.com"

begin
	# Initialize fitsspp common.
	if (firsttime) {
	    call aclri (bufpnt, NB)
	    firsttime = false
	}

	if (status > 0)
	    return

	# Convert Fortran string to an SPP string.
	call f77upk (pkname, fname, SZ_PATHNAME)

	# Check for valid unit number.
	if (funit < 1 || funit > 199) {
	    status = 101
	    return
	}

	# Find available buffer slot for this file.
	ibuff = ERR
	do i = 1, NB {
	    if (bufpnt[i] == 0) {
                 ibuff = i
                 break
	    }
	}

	# Error: no vacant buffer slots left.
	if (ibuff == ERR) {
	    status = 102
	    return
	}

	# Get the file access mode.
	if (oldnew == 0) {
	    # Test if file exists.
	    if (access (fname, 0,0) == NO) {
		# Error: file doesn't exist.
		status = 103
		return
	    }

	    # Set the access mode.
	    if (rwmode == 0)
		mode = READ_ONLY
	    else
		mode = READ_WRITE

	    # Set the FITS blocking factor.
	    block = 1
        } else
	    mode = NEW_FILE

	# Open the file.
	iferr (fd = open (fname, mode, BINARY_FILE)) {
	    if (oldnew == 0)
		status = 104
	    else
		status = 105
	    return
	}
        
	# Store the current size of the file
	filesize[ibuff] = fstati (fd, F_FILESIZE)

	# IRAF files do not have an intrinsic record length, so just use
	# the standard 2880-byte block length for convenience.

        reclen[funit] = 2880

	# Initialize FITSIO file descriptor.
	recnum[ibuff] = 0
	bytnum[ibuff] = 0
	modify[ibuff] = false
	wrmode[ibuff] = (rwmode != 0)
	chdu[ibuff] = 1
	maxhdu[ibuff] = 1
	hdstrt[ibuff,1] = 0
	hdend[ibuff] = 0
	nxthdr[ibuff] = 0

	# Data start location is undefined.
	dtstrt[ibuff] = -1000000000

	# Store internal buffer number to use for this file.
	bufnum[funit] = ibuff

	# Store inverse pointer: tells which unit is attached to this buffer.
	bufpnt[ibuff] = funit
	bufid[funit] = fd

end


# FTCLSX -- Close a file opened with FTOPNX.

procedure ftclsx (iunit, status)

int	iunit 		#I Fortran I/O unit number
int	status		#U returned error status (0=ok)

int	fd
int	ibuff
include	"fitsspp.com"

begin
	fd = bufid[iunit]
	ibuff = bufnum[iunit]

	iferr (call close(fd))
#	    set error code, if it has not previous been set
	    if (status <= 0) status = 110

	bufnum[iunit] = 0
	bufpnt[ibuff] = 0
end


# FTPI2B -- Write an array of Integer*2 bytes to the output FITS file.
#           Does any required translation from internal machine format to FITS.

procedure ftpi2b (ounit, nvals, incre, i2vals, status)

int     ounit   	#I  fortran I/O unit number
int     nvals   	#I  number of pixels in the i2vals array
int     incre   	#I  byte increment between values
short   i2vals[ARB]  	#I  array of input integer*2 values
int     status  	#U  output error status

int	i
int	offset

begin
        call miipak(i2vals,i2vals,nvals,TY_SHORT,MII_SHORT)

        if (incre .le. 2)
                call ftpbyt(ounit,nvals*2,i2vals,status)
        else   {
#               offset is the number of bytes to move between each value
                offset=incre-2
                do  i=1,nvals  {
                        call ftpbyt(ounit,2,i2vals[i],status)
                        call ftmoff(ounit,offset,true,status)                
                }
        }
end


# FTPI4B -- Write an array of Integer*4 bytes to the output FITS file.
#           Does any required translation from internal machine format to FITS.

procedure ftpi4b (ounit, nvals, incre, i4vals, status)

int     ounit   	#I  fortran I/O unit number
int     nvals   	#I  number of pixels in the i4vals array
int     incre   	#I  byte increment between values
int     i4vals[ARB]  	#I  array of input integer*4 values
int     status  	#U  output error status

int	i
int	offset

begin
        call miipak(i4vals,i4vals,nvals,TY_INT,MII_LONG)

        if (incre .le. 4)
                call ftpbyt(ounit,nvals*4,i4vals,status)
        else   {
#               offset is the number of bytes to move between each value
                offset=incre-4
                do i=1,nvals  {
                        call ftpbyt(ounit,4,i4vals[i],status)
                        call ftmoff(ounit,offset,true,status)                
                }
        }
end


# FTPR4B -- Write an array of Real*4 bytes to the output FITS file.
#           Does any required translation from internal machine format to FITS.

procedure ftpr4b (ounit, nvals, incre, r4vals, status)

int     ounit   	#I  fortran I/O unit number
int     nvals   	#I  number of pixels in the r4vals array
int     incre   	#I  byte increment between values
real    r4vals[ARB]  	#I  array of input real*4 values
int     status  	#U  output error status

int	i
int	offset

begin
        call miipak(r4vals,r4vals,nvals,TY_REAL,MII_REAL)

        if (incre .le. 4)
                call ftpbyt(ounit,nvals*4,r4vals,status)
        else   {
#               offset is the number of bytes to move between each value
                offset=incre-4
                do i=1,nvals  {
                        call ftpbyt(ounit,4,r4vals[i],status)
                        call ftmoff(ounit,offset,true,status)                
                }
        }
end


# FTPR8B -- Write an array of Real*8 bytes to the output FITS file.
#           Does any required translation from internal machine format to FITS.

procedure ftpr8b (ounit, nvals, incre, r8vals, status)

int     ounit   	#I  fortran I/O unit number
int     nvals   	#I  number of pixels in the r8vals array
int     incre   	#I  byte increment between values
double  r8vals[ARB]  	#I  array of input real*8 values
int     status  	#U  output error status

int	i
int	offset

begin
        call miipak(r8vals,r8vals,nvals,TY_DOUBLE,MII_DOUBLE)

        if (incre .le. 8)
                call ftpbyt(ounit,nvals*8,r8vals,status)
        else   {
#               offset is the number of bytes to move between each value
                offset=incre-8
                do i=1,nvals  {
                        call ftpbyt(ounit,8,r8vals[i],status)
                        call ftmoff(ounit,offset,true,status)                
                }
        }
end


# FTGI2B -- Read an array of Integer*2 bytes from the input FITS file.
#           Does any required translation from FITS to internal machine format

procedure ftgi2b (iunit, nvals, incre, i2vals, status)

int     iunit   	#I  fortran I/O unit number
int     nvals   	#I  number of pixels in the i2vals array
int     incre   	#I  byte increment between values
short   i2vals[ARB]  	#O  array of output integer*2 values
int     status  	#U  output error status

int	i
int	offset

begin
        if (incre .le. 2)
                call ftgbyt(iunit,nvals*2,i2vals,status)
        else   {
#               offset is the number of bytes to move between each value
                offset=incre-2
                do  i=1,nvals  {
                        call ftgbyt(iunit,2,i2vals[i],status)
                        call ftmoff(iunit,offset,false,status)                
                }
        }
        call miiupk(i2vals,i2vals,nvals,MII_SHORT,TY_SHORT)
end


# FTGI4B -- Read an array of Integer*4 bytes from the intput FITS file.
#           Does any required translation from FITS to internal machine format

procedure ftgi4b (iunit, nvals, incre, i4vals, status)

int     iunit   	#I  fortran I/O unit number
int     nvals   	#I  number of pixels in the i4vals array
int     incre   	#I  byte increment between values
int     i4vals[ARB]  	#O  array of output integer*4 values
int     status  	#U  output error status

int	i
int	offset

begin
        if (incre .le. 4)
                call ftgbyt(iunit,nvals*4,i4vals,status)
        else   {
#               offset is the number of bytes to move between each value
                offset=incre-4
                do  i=1,nvals  {
                        call ftgbyt(iunit,4,i4vals[i],status)
                        call ftmoff(iunit,offset,false,status)                
                }
        }
        call miiupk(i4vals,i4vals,nvals,MII_LONG,TY_INT)
end


# FTGR4B -- Read an array of Real*4 bytes from the intput FITS file.
#           Does any required translation from FITS to internal machine format

procedure ftgr4b (iunit, nvals, incre, r4vals, status)

int     iunit   	#I  fortran I/O unit number
int     nvals   	#I  number of pixels in the r4vals array
int     incre   	#I  byte increment between values
real    r4vals[ARB]  	#O  array of output real*4 values
int     status  	#U  output error status

int	i
int	offset

begin
        if (incre .le. 4)
                call ftgbyt(iunit,nvals*4,r4vals,status)
        else   {
#               offset is the number of bytes to move between each value
                offset=incre-4
                do  i=1,nvals  {
                        call ftgbyt(iunit,4,r4vals[i],status)
                        call ftmoff(iunit,offset,false,status)                
                }
        }
        call miiupk(r4vals,r4vals,nvals,MII_REAL,TY_REAL)
end


# FTGR8B -- Read an array of Real*8 bytes from the intput FITS file.
#           Does any required translation from FITS to internal machine format

procedure ftgr8b (iunit, nvals, incre, r8vals, status)

int     iunit   	#I  fortran I/O unit number
int     nvals   	#I  number of pixels in the r8vals array
int     incre   	#I  byte increment between values
double  r8vals[ARB]  	#O  array of output real*8 values
int     status  	#U  output error status

int	i
int	offset

begin
        if (incre .le. 8)
                call ftgbyt(iunit,nvals*8,r8vals,status)
        else   {
#               offset is the number of bytes to move between each value
                offset=incre-8
                do  i=1,nvals  {
                        call ftgbyt(iunit,8,r8vals[i],status)
                        call ftmoff(iunit,offset,false,status)                
                }
        }
        call miiupk(r8vals,r8vals,nvals,MII_DOUBLE,TY_DOUBLE)
end


# FTGBYT -- Read a byte sequence from a file.  The sequence may begin on any
# byte boundary and may be any number of bytes long.  An error status is
# returned if less than the requested amount of data is read.

procedure ftgbyt (iunit, nbytes, array, status)

int	iunit		#I fortran unit number
int	nbytes		#I number of bytes to be transferred
char	array[ARB]	#O output data buffer
int	status		#U output error status

int	bytes_per_record
int	fd, nbuff, fpos, nb
int	ft_readb()
include	"fitsspp.com"

begin
	# Special cases.
        if (status > 0 || nbytes == 0)
	    return
        if (nbytes < 0) {
	    status = 306
	    return
        }

	fd = bufid[iunit]

	# Get byte index in file.
        nbuff = bufnum[iunit]
	bytes_per_record = reclen[iunit]

	# zero indexed byte position in the file
	fpos = bytes_per_record * (recnum[nbuff]-1) + bytnum[nbuff]

	# Read the data.
	iferr (nb = ft_readb (fd, array, fpos, nbytes)) {
	    status = 107
	    return
	} else if (nb != nbytes) {
	    status = 107
          }

	# Update the FITSIO common to track the new file position.
	fpos = fpos + max (0, nb)


#	Note: bytnum must range from 1 to 2880 for compatibility with the
#	fortran version of FITSIO

	recnum[nbuff] = ((fpos - 1) / bytes_per_record)+1
        bytnum[nbuff] = mod ((fpos - 1), bytes_per_record) +1
end


# FTGCBF -- Read a sequence of characters from a file into the output
# character string buffer.  The sequence may begin on any byte boundary and
# may be any number of bytes long.  An error status is returned if less than
# the requested amount of data is read.

procedure ftgcbf (iunit, convrt, nbytes, array, status)

int     iunit           #I fortran unit number
int     convrt          #I convert to ASCII? (not used in SPP version)
int     nbytes          #I number of bytes to be transferred
%	character*(*)	array
int     status          #U output error status

pointer	sp
pointer	buf

begin
	call smark (sp)
	call salloc (buf, nbytes, TY_CHAR)

	# Get the data.
        call ftgbyt (iunit, nbytes, Memc[buf], status)

	if (status > 0) { 
	    call sfree (sp)
	    return
        }

	# Copy to output Fortran character string.  The only SPP callable
	# routine which writes to a Fortran character string is f77pak, so
	# we must unpack the character data into an intermediate SPP char
	# buffer first.

#  can't use f77pak because it interprets any nulls (0) as EOS and converts
#  them to blanks (ASCII 32).  Instead, call custom string copy routine.
#	call chrupk (Memc[buf], 1, Memc[buf], 1, nbytes)
#	Memc[buf+nbytes] = EOS
#	call f77pak (Memc[buf], array, nbytes)

	call fsch2c (Memc[buf], array, nbytes)

	call sfree (sp)
end

# FTPBYT -- Write a byte sequence to a file.  The sequence may begin on any
# byte boundary and may be any number of bytes long.

procedure ftpbyt (iunit, nbytes, array, status)

int	iunit		#I fortran unit number
int	nbytes		#I number of bytes to be transferred
char	array[ARB]	#I input data buffer
int	status		#U output error status

int	fd, nbuff, fpos
int	bytes_per_record
include	"fitsspp.com"

begin
	# Special cases.
        if (status > 0)
	    return
        if (nbytes < 0) {
	    status = 306
	    return
        }

	fd = bufid[iunit]

	# Get byte index in file.
        nbuff = bufnum[iunit]
	bytes_per_record = reclen[iunit]

	# zero indexed byte position in the file
	fpos = bytes_per_record * (recnum[nbuff]-1) + bytnum[nbuff]

	# nbytes=0 is a flush and reinitialize.
	if (nbytes == 0) {
	    iferr (call flush (fd))
		status = 106
	    recnum[nbuff]=0
	    bytnum[nbuff]=0
	    return
	}

	# Write the data.
	iferr (call ft_writeb (fd, array, fpos, nbytes, filesize[nbuff])) {
	    status = 107
	    return
	}

	# Update the FITSIO common to track the new file position.
	fpos = fpos + nbytes

#	Note: bytnum must range from 1 to 2880 for compatibility with the
#	fortran version of FITSIO

	recnum[nbuff] = ((fpos - 1) / bytes_per_record)+1
        bytnum[nbuff] = mod ((fpos - 1), bytes_per_record) +1

end

# FTPCBF -- Write a sequence of characters to a file.  The sequence may begin 
# on any byte boundary and may be any number of bytes long.

procedure ftpcbf (iunit, convrt, nbytes, array, status)

int	iunit		#I fortran unit number
int	convrt		#I convert to ASCII? (not used in SPP version)
int	nbytes		#I number of bytes to be transferred
%	character*(*)   array
int	status		#U output error status

pointer	sp
pointer	buf

begin
	call smark (sp)
	call salloc (buf, nbytes, TY_CHAR)

	# unpack the fortran characters into SPP CHAR
	# can't use the f77upk routine because it appends a null character
	# after the last non-blank character in the string, instead of
	# padding it out with blanks which is what we need.
	# call f77upk (array, Memc[buf], nbytes)


#  12 Jan 93: replace the following 2 calls with a simpler call to fsc2ch
#	call fsupk(array, Memc[buf], nbytes)
	# pack the SPP characters into a byte-packed array
#	call chrpak (Memc[buf], 1, Memc[buf], 1, nbytes)
 
	call fsc2ch(array, Memc[buf], nbytes)

	# Write the data.
        call ftpbyt (iunit, nbytes, Memc[buf], status)

	call sfree (sp)
end

# FSCH2C -- Copy Fortran characters from one string to another 

procedure fsch2c (a, b, nchars)

%	character*1 a(*)
%	character*(*) b
int	nchars, i

begin
	do i = 1, nchars  
%		b(i:i) = a(i)
end

# FSC2CH -- Copy Fortran characters from one string to another 

procedure fsc2ch (a, b, nchars)

%	character*(*) a
%	character*1 b(*)
int	nchars, i

begin
	do i = 1, nchars  
%		b(i) = a(i:i)
end

# FSUPK -- routine to unpack Fortran characters into an SPP CHAR array.
# This is similar to f77upk except that the CHAR string is not null-terminated.

procedure fsupk (fstr, sppchr, nchars)

%       character*(*) fstr
char sppchr[ARB]
int   nchars, i

begin
	do i = 1, nchars
%		sppchr(i)=ichar(fstr(i:i))

end

# FTPBYX -- Low-level routine to write a record to an output file (not used).

procedure ftpbyx (ounit, recnum, nbytes, nbuff, status)

int	ounit			#I fortran unit number
int	recnum			#I direct access file record number
int	nbytes			#I number of bytes to write
int	nbuff			#I number of the buffer to be written (not used)
int	status			#U output error status

begin
end


# FTGBYX -- Low-level routine to read a record from a file.
#      in the SPP version, all this does is to return an end of file
#      status if one tries to read beyond the end of file

procedure ftgbyx (ounit, numrec, nbytes, nbuff, status)

int	ounit			#I fortran unit number
int	numrec			#I direct access file record number
int	nbytes			#I number of bytes to read
int	nbuff			#I number of the buffer to be read (not used)
int	status			#U output error status

int	end_char
include	"fitsspp.com"


begin

	if (status > 0)
		return

	end_char = nbytes * numrec

	if (end_char > filesize[nbuff] * SZB_CHAR )
		status = 107

end



# FTZERO -- initialize buffer with zeros (not needed in SPP version because
#           output buffers are automatically initialized in ft_writeb).

procedure ftzero (nbuff, nwords)

int	nbuff			#I I/O buffer number
int	nwords			#I number of words to be zeroed

begin
end


# FTGSDT -- Get the current date and time.

procedure ftgsdt (dd, mm, yy, status)

int	dd		#O day of the month (1-31)
int	mm		#O month of the year (1-12)
int	yy		#O last 2 digits of the year (1992 = 92, 2001 = 01)
int	status		#U returned error status

int	itime
int	tm[LEN_TMSTRUCT]
int	clktime()

begin
	if (status > 0)
	    return

	itime = clktime (0)
	call brktime (itime, tm)

	dd = TM_MDAY(tm)
	mm = TM_MONTH(tm)
	yy = mod (TM_YEAR(tm), 100)
end


# FTUPCH -- Convert input string (a Fortran character string) to upper case.

procedure ftupch (fstr)

%       character fstr*(*)
char	sstr[SZ_LINE]

begin
	call f77upk (fstr, sstr, SZ_LINE)
	call strupr (sstr)
	call f77pak (sstr, fstr, SZ_LINE)
end


# FT_READB -- Read a sequence of bytes from a file at the indicated
# position.  The sequence can begin at any byte and can be any number of
# bytes long.
#
# This routine could be implemented more efficiently using freadp to
# directly access the file buffer for unaligned transfers, but so long
# as most transfers are aligned the following code is as fast as anything.

int procedure ft_readb (fd, obuf, fpos, nbytes)

int	fd			#I file descriptor
char	obuf[ARB]		#O output buffer
int	fpos			#I starting byte (zero index) in input file
int	nbytes			#I number of bytes to transfer

pointer	sp, bp
int	start_char, end_char
int	nchars, boff, iostat, nout
int	read()
errchk	read

begin
        # Get index of first and last file chars.
        start_char = fpos / SZB_CHAR + 1
        end_char = (fpos+nbytes  - 1) / SZB_CHAR + 1
        nchars = end_char - start_char + 1
	boff = mod (fpos, SZB_CHAR)

	# If things are nicely aligned read data directly into the output
	# buffer and we are done.

	call seek (fd, start_char)
	if (boff == 0 && mod(nbytes,SZB_CHAR) == 0)
	    return (read (fd, obuf, nchars) * SZB_CHAR)

	# Allocate intermediate buffer.
	call smark (sp)
	call salloc (bp, nchars, TY_CHAR)

	# Read raw file segment.
	iostat = read (fd, Memc[bp], nchars)
	if (iostat == EOF) {
	    call sfree (sp)
	    return (0)
	}

	# Extract and return desired bytes.
	nout = min (nbytes, iostat * SZB_CHAR - boff)
	call bytmov (Memc[bp], boff + 1, obuf, 1, nout)

	call sfree (sp)
	return (nout)
end


# FT_WRITEB -- Write a sequence of bytes to a file at the indicated
# position.  The sequence can begin at any byte and can be any number of
# bytes long.
#
# This routine could be implemented more efficiently using fwritep to
# directly access the file buffer for unaligned transfers, but so long
# as most transfers are aligned the following code is as fast as anything.

procedure ft_writeb (fd, ibuf, fpos, nbytes, fsize)

int	fd			#I file descriptor
char	ibuf[ARB]		#I data buffer
int	fpos			#I starting byte (0 index) in output file
int	nbytes			#I number of bytes to transfer
int	fsize			#I current size of the file

pointer	sp, bp, rp
int	start_char, end_char
int	nchars, boff, junk
errchk	getc, seek, write
char	getc()

begin
	call smark (sp)

        # Get index of first and last file chars.
        start_char = fpos / SZB_CHAR + 1
        end_char = (fpos+nbytes - 1) / SZB_CHAR + 1
        nchars = end_char - start_char + 1
	boff = mod (fpos, SZB_CHAR)

	# Automatically extend file by an integral number of zero-filled
	# FITS logical records if writing beyond end of file (and not
	# already appending an integral number of FITS records to the file).
	# The disk file will always be an integral number of FITS logical
	# records in length.

	if (end_char > fsize) {
	    if (!(start_char == fsize+1 &&
		boff == 0 && mod (nchars, SZ_FITSREC) == 0)) {

		call salloc (rp, SZ_FITSREC, TY_CHAR)
		call aclrc (Memc[rp], SZ_FITSREC)

		call seek (fd, fsize+1)

		while (fsize < end_char) {
		    call write (fd, Memc[rp], SZ_FITSREC)
                    fsize=fsize + SZ_FITSREC
		    }

	    }
            else
	        fsize=fsize+nchars
	}

	# If things are nicely aligned write data directly to the output
	# file and we are done.

	if (boff == 0 && mod(nbytes,SZB_CHAR) == 0) {
	    call seek (fd, start_char)
	    call write (fd, ibuf, nchars)
	    call sfree (sp)
	    return
	}

	# Allocate intermediate buffer.
	call salloc (bp, nchars, TY_CHAR)

	# Get any partial chars at ends of sequence.
	if (boff > 0) {
	    call seek (fd, start_char)
	    junk = getc (fd, Memc[bp])
	}
	if (mod (fpos+nbytes, SZB_CHAR) != 0) {
	    call seek (fd, end_char)
	    junk = getc (fd, Memc[bp+nchars-1])
	}

	# Insert data segment into buffer.
	call bytmov (ibuf, 1, Memc[bp], boff + 1, nbytes)

	# Write edited sequence to output file.
	call seek (fd, start_char)
	call write (fd, Memc[bp], nchars)

	call sfree (sp)
end
